<?php
require_once 'admin_config.php';
requireAdminAuth(); // Ensure only admins can access this page

$success = false;
$error = '';

// Handle form submission for adding a new user
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $full_name = trim($_POST['full_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $balance = floatval($_POST['balance'] ?? 0);
    $role = 'user'; // Force role to be 'user' - no new admin users can be created

    if (empty($username) || empty($email) || empty($password)) {
        $error = 'يرجى ملء الحقول المطلوبة (اسم المستخدم، البريد الإلكتروني، كلمة المرور).';
    } else if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'صيغة البريد الإلكتروني غير صحيحة.';
    } else {
        try {
            $conn = getAdminDBConnection();

            // Check if username or email already exists
            $stmt_check = $conn->prepare("SELECT id FROM users WHERE username = ? OR email = ?");
            if ($stmt_check === false) {
                throw new Exception('Prepare check statement failed: ' . $conn->error);
            }
            $stmt_check->bind_param('ss', $username, $email);
            $stmt_check->execute();
            $stmt_check->store_result();
            if ($stmt_check->num_rows > 0) {
                $error = 'اسم المستخدم أو البريد الإلكتروني موجود بالفعل.';
            } else {
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $conn->prepare("INSERT INTO users (username, full_name, email, password, balance, role, created_at) VALUES (?, ?, ?, ?, ?, ?, NOW())");
                if ($stmt === false) {
                    throw new Exception('Prepare insert statement failed: ' . $conn->error);
                }

                $stmt->bind_param('ssssds', $username, $full_name, $email, $hashed_password, $balance, $role);

                if ($stmt->execute()) {
                    $message = 'تم إضافة المستخدم بنجاح!';
                    $message_type = 'success';
                    header('Location: admin_users.php?message=' . urlencode($message) . '&type=' . $message_type);
                    exit();
                } else {
                    $error = 'حدث خطأ أثناء إضافة المستخدم: ' . $conn->error;
                }
                $stmt->close();
            }
            $stmt_check->close();
        } catch (Exception $e) {
            $error = 'خطأ: ' . $e->getMessage();
        }
    }
}

require_once '../header.php'; // Include header for consistent styling
?>

<div class="container bg-dark text-white mt-5">
    <div class="card">
        <div class="card-header">
            <h3 class="mb-0">إضافة مستخدم جديد</h3>
        </div>
        <div class="card-body">
            <?php if (!empty($message)): ?>
                <div class="alert alert-<?= htmlspecialchars($message_type) ?>" role="alert">
                    <?= htmlspecialchars($message) ?>
                </div>
            <?php elseif ($error): ?>
                <div class="alert alert-danger" role="alert">
                    <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <form method="POST">
                <div class="mb-3">
                    <label for="username" class="form-label">اسم المستخدم <span class="text-danger">*</span></label>
                    <input type="text" class="form-control" id="username" name="username" value="<?= htmlspecialchars($_POST['username'] ?? '') ?>" required>
                </div>
                <div class="mb-3">
                    <label for="full_name" class="form-label">الاسم الكامل</label>
                    <input type="text" class="form-control" id="full_name" name="full_name" value="<?= htmlspecialchars($_POST['full_name'] ?? '') ?>">
                </div>
                <div class="mb-3">
                    <label for="email" class="form-label">البريد الإلكتروني <span class="text-danger">*</span></label>
                    <input type="email" class="form-control" id="email" name="email" value="<?= htmlspecialchars($_POST['email'] ?? '') ?>" required>
                </div>
                <div class="mb-3">
                    <label for="password" class="form-label">كلمة المرور <span class="text-danger">*</span></label>
                    <input type="password" class="form-control" id="password" name="password" required>
                </div>
                <div class="mb-3">
                    <label for="balance" class="form-label">الرصيد</label>
                    <input type="number" step="0.01" class="form-control" id="balance" name="balance" value="<?= htmlspecialchars($_POST['balance'] ?? '0') ?>">
                </div>
                <div class="mb-3">
                    <label for="role" class="form-label">الدور</label>
                    <select class="form-select" id="role" name="role" disabled>
                        <option value="user" selected>مستخدم</option>
                    </select>
                    <small class="text-muted">لا يمكن إنشاء مستخدمين أدمن جدد</small>
                </div>
                <button type="submit" class="btn btn-primary">إضافة المستخدم</button>
                <a href="admin_users.php" class="btn btn-secondary">إلغاء</a>
            </form>
        </div>
    </div>
</div>

<?php require_once '../footer.php'; ?> 